document.addEventListener('DOMContentLoaded', function() {
    // Navigation functionality
    const navItems = document.querySelectorAll('.nav-item a');
    const sections = document.querySelectorAll('.section');
    
    // Function to handle navigation
    function handleNavigation(e) {
        e.preventDefault();
        const targetId = this.getAttribute('href').substring(1);
        
        // Hide all sections
        sections.forEach(section => {
            section.classList.remove('active');
        });
        
        // Show target section
        document.getElementById(targetId).classList.add('active');
        
        // Update active nav item in sidebar
        navItems.forEach(navItem => {
            navItem.parentElement.classList.remove('active');
            if (navItem.getAttribute('href') === '#' + targetId) {
                navItem.parentElement.classList.add('active');
            }
        });
        
        // Close mobile menu if open
        if (window.innerWidth < 768) {
            document.querySelector('.sidebar').classList.remove('active');
        }
    }
    
    // Add event listeners to sidebar navigation
    navItems.forEach(item => {
        item.addEventListener('click', handleNavigation);
    });
    
    // Add event listeners to all buttons with href attributes (like "View Guide" buttons)
    const allNavButtons = document.querySelectorAll('.btn[href^="#"]');
    allNavButtons.forEach(button => {
        button.addEventListener('click', handleNavigation);
    });
    
    // Accordion functionality
    const accordionItems = document.querySelectorAll('.accordion-item');
    
    accordionItems.forEach(item => {
        const header = item.querySelector('.accordion-header');
        
        header.addEventListener('click', function() {
            item.classList.toggle('active');
        });
    });
    
    // Dark mode toggle
    const themeToggle = document.querySelector('.theme-toggle');
    
    themeToggle.addEventListener('click', function() {
        document.body.classList.toggle('dark-mode');
        
        if (document.body.classList.contains('dark-mode')) {
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        } else {
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
        }
    });
    
    // Search functionality
    const searchInput = document.querySelector('.search-container input');
    const searchButton = document.querySelector('.search-container button');
    
    function performSearch() {
        const searchTerm = searchInput.value.toLowerCase();
        
        if (searchTerm.length < 2) return;
        
        let found = false;
        
        // Search in section headings and content
        sections.forEach(section => {
            const headings = section.querySelectorAll('h2, h3, h4');
            const paragraphs = section.querySelectorAll('p');
            
            let sectionMatch = false;
            
            // Check headings
            headings.forEach(heading => {
                if (heading.textContent.toLowerCase().includes(searchTerm)) {
                    sectionMatch = true;
                }
            });
            
            // Check paragraphs
            paragraphs.forEach(paragraph => {
                if (paragraph.textContent.toLowerCase().includes(searchTerm)) {
                    sectionMatch = true;
                }
            });
            
            if (sectionMatch) {
                found = true;
                
                // Show this section
                sections.forEach(s => s.classList.remove('active'));
                section.classList.add('active');
                
                // Update active nav item
                const sectionId = section.getAttribute('id');
                navItems.forEach(navItem => {
                    navItem.parentElement.classList.remove('active');
                    if (navItem.getAttribute('href') === '#' + sectionId) {
                        navItem.parentElement.classList.add('active');
                    }
                });
                
                // Highlight search terms
                const regex = new RegExp(`(${searchTerm})`, 'gi');
                paragraphs.forEach(paragraph => {
                    const originalText = paragraph.innerHTML;
                    paragraph.innerHTML = originalText.replace(regex, '<mark>$1</mark>');
                });
                
                return;
            }
        });
        
        if (!found) {
            alert('No results found for: ' + searchTerm);
        }
    }
    
    searchButton.addEventListener('click', performSearch);
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            performSearch();
        }
    });
    
    // Mobile menu toggle
    const mobileMenuToggle = document.createElement('button');
    mobileMenuToggle.classList.add('mobile-menu-toggle');
    mobileMenuToggle.innerHTML = '<i class="fas fa-bars"></i>';
    document.querySelector('.header').prepend(mobileMenuToggle);
    
    mobileMenuToggle.addEventListener('click', function() {
        document.querySelector('.sidebar').classList.toggle('active');
    });
    
    // Responsive adjustments
    function handleResize() {
        if (window.innerWidth >= 768) {
            document.querySelector('.sidebar').classList.remove('active');
        }
    }
    
    window.addEventListener('resize', handleResize);
    
    // Add mobile menu toggle button styles
    const style = document.createElement('style');
    style.textContent = `
        .mobile-menu-toggle {
            display: none;
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--dark-color);
            cursor: pointer;
            margin-right: 1rem;
        }
        
        body.dark-mode .mobile-menu-toggle {
            color: #e9ecef;
        }
        
        @media (max-width: 768px) {
            .mobile-menu-toggle {
                display: block;
            }
        }
    `;
    document.head.appendChild(style);
});